#include "xlistbox.h"
#include "xstring.h"
#include "xcombo.h"

XComboBox :: XComboBox( HWND hwnd): XListBox( hwnd )
{ ; }


/*DOC
CLASS XComboBox
FUNCTION XComboBox
GROUP constructors/destructors
REMARKS Construct a combobox

Parameters:    XWindow * theOwner        The owning window.
               XRect * rectangle         Position and size of the combobox. You must specify
                                         the size the combobox should have when the listbox is
                                         shown!
               USHORT ID                 The ID of the combobox.
                                         Default is zero.
               ULONG style               The style of the window. Valid values are:

                                             CB_SIMPLE        listbox is always shown, the user
                                                              can enter text
                                             CB_DROPDOWN      listbox is activated with special
                                                              button, the user can enter text
                                             CB_DROPDOWNLIST  listbox is activated with special
                                                              button, the user cannot enter text

                                         Default is CB_DROPDOWN.

Returns:
*/
XComboBox :: XComboBox(XWindow * owner, XRect * rec, USHORT id, ULONG style): XListBox(rec, style, owner, id, WC_COMBOBOX)
{ ;}


XListBox :: XListBox( HWND hwnd): XControl( hwnd )
{ ;}


XListBox :: XListBox(XRect * rec, ULONG style, XWindow * owner, USHORT id, PSZ className): XControl( rec, style, owner, "", className, id)
{ ;}


/*DOC
CLASS XListBox
FUNCTION XListBox
GROUP constructors/destructors
REMARKS Construct a listbox

Parameters:    XWindow * theOwner        The owning window
               XRect * rectangle         Position and size of the listbox
               USHORT ID                 The ID of the window.
                                         Default is zero.
               ULONG style               The style of the window. Valid values are:

                                             LS_SCROLL    the listbox gets a horizontal scrollbar
                                             LS_MULTI     one ore more items can be selected
                                             LS_EXTENDED  extended selection is enabled (should be
                                                          used if LS_MULTI is specified).

                                         The values can be or-ed.
Returns:
*/
XListBox :: XListBox( XWindow * owner, XRect * rec, USHORT id, ULONG style): XControl( rec, style, owner, "", WC_LISTBOX, id)
{ ;}


/*DOC
CLASS XListBox
FUNCTION InsertItem
GROUP inserting/removing
REMARKS Adds a an item to the listbox

Parameter:     char * theTitle           title of the item
               SHORT position            where to insert. You can specifythe zero-based index,
                                         LS_LAST (the item is inserted at the end of the list),
                                         LS_ASCENDING (the items are sorted ascending) or
                                         LS_DESCENDING (items are sorted descending)
                                         Default is LS_LAST.
Returns:       SHORT                     the zero-based index of the item
*/
SHORT XListBox :: InsertItem( const char * item, SHORT pos)
{
   return SHORT1FROMMR( WinSendMsg( winhandle, LM_INSERTITEM, MPFROMSHORT(pos), (PSZ) item));
}


/*DOC
CLASS XListBox
FUNCTION QueryCount
REMARKS Returns the count of items

Parameter:     -
Returns:       SHORT                      the count of items in the listbox
*/
SHORT XListBox :: QueryCount(void)
{
   return SHORT1FROMMR( WinSendMsg( winhandle,  LM_QUERYITEMCOUNT, 0, 0));
}


/*DOC
CLASS XListBox
FUNCTION QueryItemHandle
GROUP using handles
REMARKS querys a handle of an item

Parameter:     SHORT theItem             Zero-based index of the item
Returns:       LONG                      the handle of the item
*/
LONG XListBox :: QueryItemHandle( SHORT pos)
{
   return  (LONG) WinSendMsg( winhandle,  LM_QUERYITEMHANDLE, MPFROMSHORT( pos), 0) ;
}


/*DOC
CLASS XListBox
FUNCTION QueryItemText
GROUP text input/output
REMARKS Query the text of an item

Parameter:     SHORT theItem             Zero-based index of the item
               XString * buffer          buffer which will hold the text
Returns:       SHORT                     length of the text
*/
SHORT XListBox :: QueryItemText( SHORT pos, XString * buffer)
{
   SHORT r = SHORT1FROMMR(WinSendMsg( winhandle,  LM_QUERYITEMTEXTLENGTH, MPFROMSHORT( pos), 0));
   r = SHORT1FROMMR(WinSendMsg( winhandle,  LM_QUERYITEMTEXT, MPFROM2SHORT( pos, r+1), buffer->GetBuffer( r+1)));
   buffer->ReleaseBuffer();
   return r;
}


/*DOC
CLASS XListBox
FUNCTION QuerySelection
GROUP set/query selection
REMARKS Returns the zero-based index of the first selected item
which is found after the specified item

Parameters:    SHORT searchAfterItem     Zero-based index of the item behind which
                                         the search starts (defaultt is LS_FIRST)
Returns:       SHORT                     Zero-based index of the item found. If
                                         no item is found return is LS_NONE
*/
SHORT XListBox :: QuerySelection( SHORT start)
{
   return (BOOL) WinSendMsg( winhandle, LM_QUERYSELECTION, MPFROMSHORT(start), 0);
}


/*DOC
CLASS XListBox
FUNCTION RemoveAll
GROUP inserting/removing
REMARKS Removes all items from a listbox

Parameters:    -
Returns:       BOOL
*/
BOOL XListBox :: RemoveAll(void)
{
   SHORT i = InsertItem("");
   SelectItem(i);
   return SHORT1FROMMR( WinSendMsg( winhandle, LM_DELETEALL, 0, 0));
}


/*DOC
CLASS XListBox
FUNCTION RemoveItem
GROUP inserting/removing
REMARKS Removes an item

Parameter:     SHORT theItem             Zero-based index of the item to remove
Returns:       -
*/
void XListBox :: RemoveItem( SHORT pos)
{
   WinSendMsg( winhandle, LM_DELETEITEM, MPFROMSHORT(pos), 0);
}


/*DOC
CLASS XListBox
FUNCTION SearchString
REMARKS Search a string in the listbox

Parameter:     char * theText            text to be searched
               SHORT startPosition       Zero-based index of item where to start the search
               SHORT matchCode           How to search. Valid values are:

                                              LS_CASESENSITIVE  search casesensitive
                                              LS_PREFIX         search-string must be at the beginning
                                                                of the items title
                                              LS_SUBSTRING      search-string can be in the middle of
                                                                the items title
                                         LS_CASESSENSITIVE can be or-ed with one of the other defines.
Returns:       SHORT                     Zero-based index of the item which is found. If no item
                                         is found, LS_NONE is returned.
*/
SHORT XListBox :: SearchString (const char * p, SHORT start, SHORT match)
{
   return SHORT1FROMMR( WinSendMsg( winhandle, LM_SEARCHSTRING, MPFROM2SHORT(match, start), MPFROMP(p)));
}


/*DOC
CLASS XListBox
FUNCTION SelectItem
GROUP set/query selection
REMARKS Select an item

Parameter:     SHORT theItem             Zero-based index of the item to select
Returns:       -
*/
BOOL XListBox :: SelectItem( SHORT pos, BOOL select)
{
   return (BOOL) WinSendMsg( winhandle, LM_SELECTITEM, MPFROMSHORT(pos), MPFROMSHORT(select));
}


/*DOC
CLASS XListBox
FUNCTION SetItemHandle
GROUP using handles
REMARKS Adds a handle to an item

Parameter:     SHORT theItem             Zero-based index of the item which
                                         will get the handle
               LONG theHandle            the handle which is add to the item
Returns:       BOOL                      success
*/
BOOL XListBox :: SetItemHandle(SHORT pos, LONG handle)
{
   return (BOOL) WinSendMsg( winhandle, LM_SETITEMHANDLE, MPFROMSHORT(pos), (MPARAM) handle);
}


/*DOC
CLASS XListBox
FUNCTION SetItemText
GROUP text input/output
REMARKS Set the text of an item

Parameter:     SHORT theItem             Zero-based index of the item
               char * theText            the new text
Returns:       BOOL                      success
*/
BOOL XListBox :: SetItemText(SHORT pos, const char * p)
{
   return SHORT1FROMMR( WinSendMsg( winhandle, LM_SETITEMTEXT, MPFROMSHORT(pos), MPFROMP(p)));
}

