/*
 * REGISTER.C
 *
 * Functions to perform enumerations and lookups into the registration
 * database for the use of OLE client applications.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 */

#include <windows.h>
#include <shellapi.h>
#include "register.h"


/*
 * WFillClassList
 *
 * Purpose:
 *  Enumerates available OLE object classes from the registration
 *  database and fills a listbox with those names.
 *
 *  Note that this function removes any prior contents of the listbox.
 *
 * Parameters:
 *  hList           HWND to the listbox to fill.
 *
 * Return Value:
 *  WORD            Number of strings added to the listbox, -1 on failure.
 */

WORD FAR PASCAL WFillClassList(HWND hList)
    {
    DWORD       dw;
    WORD        cStrings;
    HKEY        hKey;
    LONG        lRet;
    char        szExec[CBMAXKEY];
    char        szClass[CBMAXKEY];
    char        szKey[CBMAXKEY];

    //Clean out the existing strings.
    SendMessage(hList, LB_RESETCONTENT, 0, 0L);

    //Open up the root key.
    lRet=RegOpenKey(HKEY_CLASSES_ROOT, NULL, &hKey);

    if ((LONG)ERROR_SUCCESS!=lRet)
        return (WORD)-1;

    cStrings=0;
    lRet=RegEnumKey(hKey, cStrings++, szClass, CBMAXKEY);

    while ((LONG)ERROR_SUCCESS==lRet)
        {
        //Check for a \protocol\StdFileEditing\server entry.
        lstrcpy(szExec, szClass);
        lstrcat(szExec, "\\protocol\\StdFileEditing\\server");

        dw=CBMAXKEY;

        lRet=RegQueryValue(hKey, szExec, szKey, &dw);

        if ((LONG)ERROR_SUCCESS==lRet)
            {
            dw=CBMAXKEY;

            /*
             * If we have an .EXE, get the class name and send it to
             * the listbox.
             */
            lRet=RegQueryValue(hKey, szClass, szKey, &dw);

            if ((LONG)ERROR_SUCCESS==lRet)
                SendMessage(hList, LB_ADDSTRING, 0, (DWORD)(LPSTR)szKey);
            }

        //Continue with the next key.
        lRet=RegEnumKey(hKey, cStrings++, szClass, CBMAXKEY);
        }

    RegCloseKey(hKey);
    return cStrings;
    }




/*
 * WClassFromDescription
 *
 * Purpose:
 *  Looks up the actual OLE class name in the registration database
 *  for the given descriptive name chosen from a listbox.
 *
 * Parameters:
 *  psz             LPSTR to the descriptive name.
 *  pszClass        LPSTR in which to store the class name.
 *  cb              WORD maximum length of pszClass.
 *
 * Return Value:
 *  WORD            Number of characters copied to pszClass.  0 on failure.
 */

WORD FAR PASCAL WClassFromDescription(LPSTR psz, LPSTR pszClass, WORD cb)
    {
    DWORD           dw;
    HKEY            hKey;
    char            szClass[CBMAXKEY];
    LONG            lRet;
    WORD            i;

    //Open up the root key.
    lRet=RegOpenKey(HKEY_CLASSES_ROOT, NULL, &hKey);

    if ((LONG)ERROR_SUCCESS!=lRet)
        return 0;

    i=0;
    lRet=RegEnumKey(hKey, i++, szClass, CBMAXKEY);

    while ((LONG)ERROR_SUCCESS==lRet)
        {
        dw=(DWORD)cb;
        lRet=RegQueryValue(hKey, szClass, pszClass, &dw);

        if ((LONG)ERROR_SUCCESS==lRet)
            {
            if (!lstrcmp(pszClass, psz))
                break;
            }

        //Continue with the next key.
        lRet=RegEnumKey(hKey, i++, szClass, CBMAXKEY);
        }

    if ((LONG)ERROR_SUCCESS==lRet)
        lstrcpy(pszClass, szClass);
    else
        dw=0L;

    RegCloseKey(hKey);
    return (WORD)dw;
    }



/*
 * WClassFromExtension
 *
 * Purpose:
 *  Looks up the OLE class name name in the registration database for
 *  the given file extension.
 *
 * Parameters:
 *  pszExt          LPSTR to the extension name.
 *  psz             LPSTR in which to store the class name.
 *  cb              WORD maximum length of the class name.
 *
 * Return Value:
 *  WORD            Number of characters copied to pszClass.  0 on failure.
 */

WORD FAR PASCAL WClassFromExtension(LPSTR pszExt, LPSTR psz, WORD cb)
    {
    DWORD           dw;
    HKEY            hKey;
    LONG            lRet;

    if (NULL==pszExt || NULL==psz)
        return 0;

    //Open up the root key.
    lRet=RegOpenKey(HKEY_CLASSES_ROOT, NULL, &hKey);

    if ((LONG)ERROR_SUCCESS!=lRet)
        return 0;

    //Get the class name using the extension.
    dw=(DWORD)cb;
    lRet=RegQueryValue(hKey, pszExt, psz, &dw);

    RegCloseKey(hKey);

    if ((LONG)ERROR_SUCCESS!=lRet)
        return 0;

    return (WORD)dw;
    }









/*
 * CVerbEnum
 *
 * Purpose:
 *  Builds a double-null terminated list of verbs for a particular class.
 *
 * Parameters:
 *  pszClass        LPSTR to the object classname.
 *  pszVerbs        LPSTR in which to store the verbs.
 *  cbMax           WORD maximum number of characters in pszVerbs.
 *
 * Return Value:
 *  WORD            Number of verbs stored.
 */

WORD FAR PASCAL CVerbEnum(LPSTR pszClass, LPSTR pszVerbs, WORD cbMax)
    {
    DWORD           dw;
    WORD            cb;
    HKEY            hKey;
    HKEY            hKeyVerb;
    LONG            lRet;
    char            szVerbNum[10];
    HANDLE          hMem=NULL;
    WORD            i;

    if (NULL==pszClass || NULL==pszVerbs || 0==cbMax)
        return NULL;

    /*
     * Open up the key for <pszClass>\\protocol\\StdFileEditing\\verb
     * We first open up the pszClass key from which we can open a subkey
     * of the \\protocol\\StdFileEditing\\verb, which saves us from
     * having to prepend that mess to each verb number.
     */
    lRet=RegOpenKey(HKEY_CLASSES_ROOT, pszClass, &hKey);

    if ((LONG)ERROR_SUCCESS!=lRet)
        return 0;

    lRet=RegOpenKey(hKey, "protocol\\StdFileEditing\\verb", &hKeyVerb);

    //We will not need this key any more.
    RegCloseKey(hKey);

    if ((LONG)ERROR_SUCCESS!=lRet)
        return 0;


    //Start at verb 0
    i=0;
    cb=0;

    /*
     * Extract verbs from the registration database directly into
     * the list.  cbMax is decremented each time by the number of
     * character stored in the memory.
     */

    while ((LONG)ERROR_SUCCESS==lRet && cbMax > 0)
        {
        /*
         * These operations make up for the previous RegQueryValue call.
         * cb includes the null terminator.
         */
        pszVerbs+=(DWORD)cb;
        cbMax-=cb;
        cb=cbMax;

        wsprintf(szVerbNum, "%d", i++);
        lRet=RegQueryValue(hKeyVerb, szVerbNum, pszVerbs, &dw);
        cb=(WORD)dw;
        }

    //Null-terminate the list.
    *pszVerbs=0;

    RegCloseKey(hKeyVerb);

    /*
     * Return the number of verbs we added.  i will be one greater since
     * it was incremented the last time we failed RegQueryValue.
     */

    return --i;
    }




/*
 * WDescriptionFromClass
 *
 * Purpose:
 *  Looks up the actual OLE descriptive name name in the registration
 *  database for the given class name.
 *
 * Parameters:
 *  pszClass        LPSTR to the class name.
 *  psz             LPSTR in which to store the descriptive name.
 *  cb              WORD maximum length of psz.
 *
 * Return Value:
 *  WORD            Number of characters copied to pszClass.  0 on failure.
 */

WORD FAR PASCAL WDescriptionFromClass(LPSTR pszClass, LPSTR psz, WORD cb)
    {
    DWORD           dw;
    HKEY            hKey;
    LONG            lRet;

    if (NULL==pszClass || NULL==psz)
        return 0;

    //Open up the root key.
    lRet=RegOpenKey(HKEY_CLASSES_ROOT, NULL, &hKey);

    if ((LONG)ERROR_SUCCESS!=lRet)
        return 0;

    //Get the descriptive name using the class name.
    dw=(DWORD)cb;
    lRet=RegQueryValue(hKey, pszClass, psz, &dw);

    RegCloseKey(hKey);

    psz+=lstrlen(psz)+1;
    *psz=0;

    if ((LONG)ERROR_SUCCESS!=lRet)
        return 0;

    return (WORD)dw;
    }
