/*
 * OLESTREA.C
 *
 * Constructor, Destructor, and Methods for the STREAM structure.  The
 * methods are called from OLECLI.DLL through the LPOLESTREAMVTBL at
 * the beginning of the STREAM structure.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 *
 */

#include <windows.h>
#include <ole.h>
#include "oclient.h"



/*
 * PStreamAllocate
 *
 * Purpose:
 *  Allocates and initializes a STREAM structure, using the function
 *  PVtblStreamAllocate to initialize its VTBL.  If pfnGet is NULL,
 *  we use the StreamGet method in OLESTREA.C.  Likewise, if pfnPut is
 *  NULL, we use the StreamPut method in OLESTREA.C.
 *
 * Parameters:
 *  pfSuccess       LPBOOL indicating if the initialization succeeded.  If
 *                  this function returns non-NULL, but *pfSuccess==FALSE,
 *                  the caller must call the destructor function.
 *  hInst           HANDLE of the application instance.
 *  pfnGet          FARPROC to the stream's Get method.
 *  pfnPut          FARPROC to the stream's Put method.
 *
 * Return Value:
 *  LPSTREAM        Pointer to the allocated STREAM if successful, NULL
 *                  if the allocation failed or a parameter is invalid.
 */

LPSTREAM FAR PASCAL PStreamAllocate(LPBOOL pfSuccess, HANDLE hInst,
                                    FARPROC pfnGet, FARPROC pfnPut)
    {
    HANDLE          hMem;
    LPSTREAM        pStream;

    if (NULL==pfSuccess)
        return NULL;

    *pfSuccess=FALSE;

    if (NULL==pfnGet)
        pfnGet=(FARPROC)StreamGet;

    if (NULL==pfnPut)
        pfnGet=(FARPROC)StreamPut;


    hMem=LocalAlloc(LPTR, CBSTREAM);

    if (NULL==hMem)
        return FALSE;

    //All fields are initially NULL from the LocalAlloc.
    pStream=(LPSTREAM)(PSTR)hMem;

    //Get initialized OLESTREAMVTBL pointer.  Note that this sets pfSuccess.
    pStream->pvt=PVtblStreamAllocate(pfSuccess, hInst, pfnGet, pfnPut);
    return pStream;
    }



/*
 * PStreamFree
 *
 * Purpose:
 *  Frees all data in the STREAM and frees the structure.
 *
 * Parameters:
 *  pStream         LPSTREAM to the structure to free.
 *
 * Return Value:
 *  LPSTREAM        NULL if the function succeeds, pStream otherwise.
 */

LPSTREAM FAR PASCAL PStreamFree(LPSTREAM pStream)
    {
    if (NULL==pStream)
        return NULL;

    //Free this object's VTBL
    if (NULL!=PVtblStreamFree(pStream->pvt))
        return pStream;

    if (NULL!=LocalFree((HANDLE)(DWORD)pStream))
        return pStream;

    return NULL;
    }








/*
 * StreamGet
 *
 * Purpose:
 *  Instructs the client to read a specified number of bytes from its
 *  already open file.  This method must be capable of handling data that
 *  is greater than 64K, which is why we use the DwReadHuge function
 *  in fileio.c.
 *
 * Parameters:
 *  pStream         LPSTREAM to the stream structure holding the file handle.
 *  pb              LPBYTE in which to read the data.  We have no idea what
 *                  data we'll read since it's native server data.
 *  cb              DWORD number of bytes to read from the file into pb.
 *
 * Return Value:
 *  DWORD           Number of bytes actually read.
 */

DWORD FAR PASCAL StreamGet(LPSTREAM pStream, LPBYTE pb, DWORD cb)
    {
    DWORD       cbRead;

    /*
     * If you already have a file handle in the STREAM structure,
     * then just read cb bytes of the data into pb from that file handle.
     * This assumes that we are in the process of reading a file and
     * have an open file handle.
     */

    if (NULL==pStream->hFile)
        return 0L;

    cbRead=DwReadHuge(pStream->hFile, (LPVOID)pb, cb);

    //If cb!=cbRead, OLECLI will consider it an error.  No need to check here.
    return cbRead;
    }




/*
 * StreamPut
 *
 * Purpose:
 *  Instructs the client to write a specified number of bytes to its
 *  already open file.  This method must be capable of handling data that
 *  is greater than 64K, which is why we use the DwWriteHuge function
 *  in fileio.c.
 *
 * Parameters:
 *  pStream         LPSTREAM to the stream structure holding the file handle.
 *  pb              LPBYTE to the data to write.  We have no idea what
 *                  data sits here.
 *  cb              DWORD number of bytes to write from pb.
 *
 * Return Value:
 *  DWORD           Number of bytes actually written.
 */

DWORD FAR PASCAL StreamPut(LPSTREAM pStream, LPBYTE pb, DWORD cb)
    {
    DWORD       cbWritten;

    /*
     * If you already have a file handle in the STREAM structure,
     * then just write cb bytes of the data at pb to that file handle.
     * This assumes that we are in the process of writing a file and
     * have an open file handle.
     */

    if (NULL==pStream->hFile)
        return 0L;

    cbWritten=DwWriteHuge(pStream->hFile, (LPVOID)pb, cb);

    //If cb!=cbWritten, OLECLI will consider it an error.  No need to check here.
    return cbWritten;
    }
