/*
 * OLEMENU.C
 *
 * Functions to enable or disable OLE menu items as well as
 * menu items depending on the state of the clipboard.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 *
 */

#include <windows.h>
#include <ole.h>
#include "register.h"
#include "oclient.h"



/*
 * MenuOLEClipboardEnable
 *
 * Purpose:
 *  Enabled or disables Edit menu commands for Paste, PasteLink, Paste
 *  Special, and Links, depending on whether or not certain clipboard
 *  formats exists or if there are linked objects.
 *
 *  This function should be called for the WM_INITPOPUPMENU message in the
 *  main window procedure.
 *
 * Parameters:
 *  hMenu           HMENU of the Edit popup menu.
 *  pDoc            LPDOCUMENT owner of all objects.
 *  pWID            LPWORD to an array of four WORDs, where the caller
 *                  stores menu ID values for Paste, PasteLink, PasteSpecial,
 *                  and Links.  If any of these are zero that menu item is
 *                  ignored.
 *
 * Return Value:
 *  None
 */

void FAR PASCAL MenuOLEClipboardEnable(HMENU hMenu, LPDOCUMENT pDoc, LPWORD pWID)
    {
    OLESTATUS   os;
    BOOL        fEmbed;
    BOOL        fLink;
    BOOL        fStatic;
    WORD        wTemp;

    if (NULL==pWID)
        return;

    /*
     * Check if we can create an embedded, linked, or static object
     * from the clipboard.  Note that olerender_draw means that fourth
     * parameter to OleQueryCreateFromClip is ignored.
     */

    os=OleQueryCreateFromClip(PSZOLE(IDS_STDFILEEDITING), olerender_draw, 0);
    fEmbed=(OLE_OK==os);

    //Check if we can create a linked object from the clipboard.
    os=OleQueryLinkFromClip(PSZOLE(IDS_STDFILEEDITING), olerender_draw, 0);
    fLink=(OLE_OK==os);

    os=OleQueryCreateFromClip(PSZOLE(IDS_STATIC), olerender_draw, 0);
    fStatic=(OLE_OK==os);


    /*
     * Enable menu items depending on the available formats:
     *  Paste:          If fEmbed or fStatic is TRUE
     *  Paste Link:     Only if fLink if TRUE.
     *  Paste Special:  If fEmbed, fLink, or fStatic is TRUE.
     */

    if (0!=pWID[0])
        {
        wTemp=(fEmbed | fStatic) ? MF_ENABLED : (MF_DISABLED | MF_GRAYED);
        EnableMenuItem(hMenu, pWID[0], wTemp | MF_BYCOMMAND);
        }

    if (0!=pWID[1])
        {
        wTemp=(fLink) ? MF_ENABLED : (MF_DISABLED | MF_GRAYED);
        EnableMenuItem(hMenu, pWID[1], wTemp | MF_BYCOMMAND);
        }

    if (0!=pWID[2])
        {
        wTemp=(fEmbed | fLink | fStatic) ? MF_ENABLED : (MF_DISABLED | MF_GRAYED);
        EnableMenuItem(hMenu, pWID[2], wTemp | MF_BYCOMMAND);
        }


    /*
     * Enumerate the objects--if we encounter any that IS linked,
     * the FEnumOLEPaste function returns FALSE, so FObjectsEnumerate
     * returns FALSE, which tells us that we HAVE a linked object.
     * Only if FObjectsEnumerate returns TRUE do we disable the item.
     */

    if (0!=pWID[3])
        {
        fLink=!FObjectsEnumerate(pDoc, FEnumOLEPaste, 0L);
        wTemp=(fLink) ? MF_ENABLED : (MF_DISABLED | MF_GRAYED);
        EnableMenuItem(hMenu, pWID[3], wTemp | MF_BYCOMMAND);
        }

    return;
    }





/*
 * FEnumOLEPaste
 *
 * Purpose:
 *  Checks each enumerated object for being of type OT_LINK. If
 *  any object is, we return FALSE to say we found one.  We otherwise
 *  return TRUE.  Utterly simple function.
 *
 * Parameters:
 *  pDoc            LPDOCUMENT to the owner of the objects.
 *  pObj            LPOBJECT of the object to delete.
 *  dw              DWORD for extra data, unused.
 *
 * Return Value:
 *  BOOL            TRUE, indicates to continue enumeration.
 */

BOOL FAR PASCAL FEnumOLEPaste(LPDOCUMENT pDoc, LPOBJECT pObj, DWORD dw)
    {
    return (OT_LINK!=pObj->dwType);
    }




/*
 * MenuOLEVerbAppend
 *
 * Purpose:
 *  Appends the appropriate menu item(s) on the given menu depending
 *  on the given object.  This function finds the verbs for the given
 *  object and if there's one, it creates a single menu item with
 *  <verb><object>.  If there's multiple verbs, it creates a cascading
 *  menu with <object> > <verb 0>
 *                       <verb 1>
 *                         ...
 *
 *  Call this function when processing the WM_INITPOPUPMENU message.
 *
 *  This function requires HVerbEnum function in register.c.
 *
 * Parameters:
 *  hMenu           HMENU of the Edit popup menu.
 *  iVerbMenu       WORD position of the item to modify.
 *  wIDMin          WORD first menu ID value for the menu items.
 *  pDoc            LPDOCUMENT owner of the object which contains clipboard
 *                  formats.
 *  pObj            LPOLEOBJECT pointing to the object used to enumerate
 *                  the verbs to append to the menu.
 *
 * Return Value:
 *  None
 */

void FAR PASCAL MenuOLEVerbAppend(HMENU hMenu, WORD iVerbMenu, WORD wIDMin,
                                  LPDOCUMENT pDoc, LPOBJECT pObj)
    {
    LPSTR               pszClass;
    LPSTR               pszVerbs;
    WORD                cf;
    WORD                cVerbs;
    HMENU               hMenuT;

    if (NULL==hMenu)
        return;

    //Delete whatever menu is in the position we're going to modify.
    DeleteMenu(hMenu, iVerbMenu, MF_BYPOSITION);

    if (NULL==pDoc)
        return;

    /*
     * If there's no object, then make this menu read "Object" and
     * disable and gray it.
     */
    if (NULL==pObj)
        {
        InsertMenu(hMenu, iVerbMenu, MF_DISABLED | MF_GRAYED | MF_STRING | MF_BYPOSITION,
                   wIDMin, PSZOLE(IDS_OBJECT));

        return;
        }

    //Retrieve the class name for the object.
    cf=(OT_LINK==pObj->dwType) ? pDoc->cfObjectLink : pDoc->cfOwnerLink;

    if (!FObjectDataGet(pObj, cf, pDoc->pszData1))
        {
        //Add the default menu item.
        InsertMenu(hMenu, iVerbMenu, MF_DISABLED | MF_GRAYED | MF_STRING | MF_BYPOSITION,
                   wIDMin, PSZOLE(IDS_OBJECT));
        return;
        }


    /*
     * Go get the list of verbs for this object; exit if we got no list or
     * no verbs.  We can pass pszClass directly even if it's NULL since
     * hVerbEnum will fail on NULL==pszClass.
     *
     * If there are no verbs or CVerbEnum fails, then default to "Edit."
     * Some old Windows apps register no verbs, so you need to prepare
     * a default.
     */

    pszClass=pDoc->pszData1;
    pszVerbs=pDoc->pszData2;
    cVerbs=CVerbEnum(pszClass, pszVerbs, CBSCRATCH);

    if (0==cVerbs)
        {
        cVerbs=1;
        lstrcpy(pszVerbs, PSZOLE(IDS_VERBEDIT));
        }

    //Change the classname to a descriptive name.
    lstrcpy(pDoc->pszData3, pszClass);
    WDescriptionFromClass(pDoc->pszData3, pszClass, CBSCRATCH);

    /*
     * Create a new popup menu and fill it with verbs, if we have more
     * than one verb.  Otherwise add a single menu items in the form
     * of <verb-0><object>.
     */
    if (1==cVerbs)
        {
        wsprintf(pDoc->pszData3, "%s %s &Object", pszVerbs, pszClass);
        InsertMenu(hMenu, iVerbMenu, MF_STRING | MF_BYPOSITION, wIDMin, pDoc->pszData3);
        }
    else
        {
        //Add a popup menu the main menu.
        hMenuT=CreatePopupMenu();

        wsprintf(pDoc->pszData3, "%s &Object", pszClass);
        InsertMenu(hMenu, iVerbMenu, MF_STRING | MF_POPUP | MF_BYPOSITION,
                   hMenuT, pDoc->pszData3);

        //Add all the verbs to the popup menu.
        while (*pszVerbs)
            {
            InsertMenu(hMenuT, -1, MF_STRING | MF_BYPOSITION, wIDMin++, pszVerbs);
            pszVerbs+=lstrlen(pszVerbs)+1;
            }
        }

    return;
    }
