/*
 * OLELOAD.C
 *
 * Functions to handle updating links on File Open.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 */


#include <windows.h>
#include <ole.h>
#include "register.h"
#include "oclient.h"



/*
 * FObjectAutoLinkUpdate
 *
 * Purpose:
 *  Checks if the object link is automatic and if so, update it if the
 *  server is open, waiting as necessary.
 *
 * Parameters:
 *  pDoc            LPDOCUMENT containing OLE information.
 *  pObj            LPOBJECT to the object in question.
 *
 * Return Value:
 *  BOOL            TRUE if the object was updated, FALSE if the object
 *                  is a manual link or the server was not open.
 */

BOOL FAR PASCAL FObjectAutoLinkUpdate(LPDOCUMENT pDoc, LPOBJECT pObj)
    {
    OLESTATUS       os;

    if (NULL==pObj)
        return FALSE;

    //Check for an automatic link.
    if (oleupdate_always!=pObj->dwLink)
        return FALSE;

    /*
     * Check if the server is open for this object with OleQueryOpen.  If
     * it is, then call OleUpdate and wait for the update to complete
     * before moving on.
     */
    os=OleQueryOpen(pObj->pObj);

    if (OLE_OK!=os)
        return FALSE;

    os=OleUpdate(pObj->pObj);

    return (OLE_OK==OsError(os, pDoc, pObj, TRUE));
    }






/*
 * FOLELinksUpdate
 *
 * Purpose:
 *  Checks if the recently loaded file had objects requiring update.
 *  If so, then prompt the user to update links, and if they answer
 *  Yes, then call OleUpdate for all linked objects.
 *
 *  If any of the links cannot be updated, then we prompt the user and
 *  fall into the Links dialog if they so choose.
 *
 * Parameters:
 *  hWnd            HWND parent of dialogs.
 *  hInst           HANDLE to the application instance.
 *  pDoc            LPDOCUMENT identifying the list of objects we have.
 *
 * Return Value:
 *  BOOL            TRUE if all objects could be updated or if the user
 *                  used the Links dialog.  FALSE if there are still
 *                  non-updated links.
 */

BOOL FAR PASCAL FOLELinksUpdate(HWND hWnd, HANDLE hInst, LPDOCUMENT pDoc)
    {
    FARPROC         lpfn;
    WORD            wTemp;

    //Check if we have any links to update
    if (0==pDoc->cLinks)
        return TRUE;

    //Ask the user.
    wTemp=MessageBox(hWnd, PSZOLE(IDS_UPDATELINKS0), PSZOLE(IDS_UPDATEMSG),
                     MB_YESNO | MB_ICONEXCLAMATION);


    if (IDYES!=wTemp)
        return FALSE;

    pDoc->cLinks=0;
    FObjectsEnumerate(pDoc, FEnumUpdate, 0L);
    FOLEReleaseWait(TRUE, pDoc, NULL);


    /*
     * If pDoc->cLinks!=0 now, then we had some unavailable links
     * which we now need to let the user update, if desired.
     */
    if (0==pDoc->cLinks)
        return TRUE;

    //Show the dialog allowing transit to the Links... dialog.
    lpfn=MakeProcInstance(LinksUnavailableProc, hInst);
    wTemp=DialogBox(hInst, MAKEINTRESOURCE(IDD_BADLINKS), hWnd, lpfn);
    FreeProcInstance(lpfn);

    if (0!=wTemp)
        return FALSE;

    //Go do the links dialog
    return FOLELinksEdit(hWnd, hInst, pDoc);
    }




/*
 * FEnumUpdate
 *
 * Purpose:
 *  For each enumerated object that is linked call OleUpdate.  We expect
 *  to be used from FOLELinksUpdate which is used when opening a document.
 *
 * Parameters:
 *  pDoc            LPDOCUMENT identifying the owner of all objects.
 *  pObj            LPOBJECT identifying the current object.
 *  dw              DWORD for extra data, unused.
 *
 * Return Value:
 *  BOOL            TRUE--we want to enumerate everything.
 */

BOOL FAR PASCAL FEnumUpdate(LPDOCUMENT pDoc, LPOBJECT pObj, DWORD dw)
    {
    OLESTATUS       os;

    pDoc->cWait=0;

    if (OT_LINK==pObj->dwType)
        {
        os=OleUpdate(pObj->pObj);

        /*
         * Wait for this object now since we need to know if it succeeded
         * or not after the object is released.
         */
        if (OLE_OK!=OsError(os, pDoc, pObj, TRUE))
            {
            //Mark this link as Unavailable.
            pObj->dwLink=OLEUPDATE_UNAVAILABLE;
            pDoc->cLinks++;
            }
        }


    return TRUE;
    }




/*
 * LinksUnavailableProc
 *
 * Purpose:
 *  Dialog procedure for the dialog to inform the user that some
 *  linked files were unavailable, giving the user the option to
 *  change the links with the Links dialog.
 *
 *  The only reason we have this dialog is to enable us to use a
 *  system icon and a non-standard button.  Otherwise we could use
 *  a message box.
 *
 *  On return from this function, the caller should check the return
 *  value.  If it's 0, then the caller should display the Links
 *  dialog.
 *
 * Parameters:
 *  The standard.
 *
 * Return Value:
 *  The value to be returned through the DialogBox call that
 *  created the dialog.
 */

BOOL FAR PASCAL LinksUnavailableProc(HWND hDlg, WORD iMsg, WORD wParam, LONG lParam)
    {
    HANDLE      hIcon;

    switch (iMsg)
        {
        case WM_INITDIALOG:
            //No need to free the previous icon since there wasn't one
            hIcon=LoadIcon(NULL, MAKEINTRESOURCE(IDI_EXCLAMATION));

            //Note:  This is Windows 3.1 specific.  Use SetDlgItemText in 3.0
            SendDlgItemMessage(hDlg, ID_ICONEX, STM_SETICON, hIcon, 0L);
            return TRUE;

        case WM_COMMAND:
            switch (wParam)
                {
                //Note that we do not want to delete the system icon.

                case ID_LINKS:
                    EndDialog(hDlg, FALSE);
                    break;

                case IDOK:
                    EndDialog(hDlg, TRUE);
                    break;
                }
            break;
        }
    return FALSE;
    }
