/*
 * OLELINK2.C
 *
 * Helper functions for the Links dialog:
 *  EnableLinkButtons:    Enables or disables links dialog buttons
 *                        depending on the selections.
 *
 *  CchLinkStringCreate:  Parses ObjectLink data and creates a string
 *                        of the object name, link file, selection, and
 *                        update status, separated by tabs.
 *
 *  LinkStringChange:     Replaces one string in a listbox with another,
 *                        preserving item data and selection state.
 *
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 */


#include <windows.h>
#include <ole.h>
#include "register.h"
#include "oclient.h"


extern WORD  cxList;
extern HWND  hgList;



/*
 * EnableLinkButtons
 *
 * Purpose:
 *  Enables or disables various buttons in the Links dialog as appropriate
 *  for what exists and is selected.
 *
 * Parameters:
 *  hDlg            HWND of the Links dialog box.
 *  hList           HWND of the listbox with link items.
 *
 * Return Value:
 *  None
 */

void FAR PASCAL EnableLinkButtons(HWND hDlg, HWND hList)
    {
    LPOBJECT        pObj;
    WORD            i;
    ATOM            aCurLink=0;
    WORD            cAuto=0;
    WORD            cManual=0;
    WORD            cStatic=0;
    WORD            cUnavailable=0;
    WORD            cLinks;
    BOOL            fAuto;
    BOOL            fManual;
    BOOL            fUnavailable;
    BOOL            fNone;
    BOOL            fAutoOnly;
    BOOL            fManualOnly;
    BOOL            fUnavailableOnly;
    BOOL            fUpdate;
    BOOL            fChangeLink=TRUE;


    cLinks=(WORD)SendMessage(hList, LB_GETCOUNT, 0, 0L);

    /*
     * Loop through the items and count which ones are manual,
     * automatic, and static.  I don't use FLinksEnumerate here because
     * we have to affect so many variables.  The flow of control is not
     * difficult here since we have no errors.
     */
    for (i=0; i < cLinks; i++)
        {
        //Check if this item is selected.
        if (SendMessage(hList, LB_GETSEL, i, 0L))
            {
            //Get the data pointer for this item.
            pObj=(LPOBJECT)SendMessage(hList, LB_GETITEMDATA, i, 0L);

            if (OT_STATIC==pObj->dwType)
                cStatic++;
            else
                {
                switch(pObj->dwLink)
                    {
                    case oleupdate_always:
                        cAuto++;
                        break;

                    case oleupdate_oncall:
                        cManual++;
                        break;

                    case OLEUPDATE_UNAVAILABLE: //App-defined.
                        cUnavailable++;
                        break;
                    }

                /*
                 * As we go through, check if each link selected is from
                 * the same file.  We use pObj->aLink to compare filenames.
                 *
                 * If we encounter any non-matching link pathname, then
                 * the Change Links button is diabled.
                 */

                if (0==aCurLink)
                   aCurLink=pObj->aLink;
                else
                    {
                    if (aCurLink!=pObj->aLink)
                        fChangeLink=FALSE;
                    }
                }
            }
        }

    /*
     * Disable all buttons if there are no manual or automatic links
     * selected.  In addition, unselect both Update radiobuttons.
     */

    fAuto           =(0!=cAuto);
    fManual         =(0!=cManual);
    fUnavailable    =(0!=cUnavailable);
    fNone           =(0==cLinks || !(fAuto || fManual || fUnavailable));
    fAutoOnly       =(fAuto && !fManual);
    fManualOnly     =(fManual && !fAuto);
    fUnavailableOnly=((0!=cUnavailable) && !fAuto && !fManual);

    fChangeLink = (fChangeLink || fUnavailableOnly) && !fNone;

    //If there are any links selected, we can update or cancel them
    fUpdate=(fAuto || fManual) && !fUnavailable;
    EnableWindow(GetDlgItem(hDlg, ID_UPDATENOW),  !fNone && fUpdate);
    EnableWindow(GetDlgItem(hDlg, ID_CANCELLINK), !fNone && (fAuto || fManual));

    //If there are links or all links are from the same file, enable Change Link
    EnableWindow(GetDlgItem(hDlg, ID_CHANGELINK), fChangeLink);

    /*
     * Clear selections in the Update radiobuttons if there's no links or
     * the links are not all the same type.
     */

    CheckDlgButton(hDlg, ID_UPDATEAUTO,   !fNone && fAutoOnly);
    CheckDlgButton(hDlg, ID_UPDATEMANUAL, !fNone && fManualOnly);

    /*
     * If there are no automatic or manual links selected then disable
     * the update buttons, otherwise reenable them.
     */
    EnableWindow(GetDlgItem(hDlg, ID_UPDATEAUTO),   !fNone);
    EnableWindow(GetDlgItem(hDlg, ID_UPDATEMANUAL), !fNone);

    return;
    }








/*
 * CchLinkStringCreate
 *
 * Purpose:
 *  Creates a Links... listbox string from an linked object.  The
 *  ObjectLink data is stored in three ATOMs in this object which we
 *  created in PObjectAllocate (OLEOBJ.C).  We also append the type of link
 *  (Automatic, Manual, or Unavailable) to the string.
 *
 *  Each string is visually limited to a tab space in the listbox.
 *
 * Parameters:
 *  psz             LPSTR pointing to the buffer to receive the string.
 *  pDoc            LPDOCUMENT containing OLE information.
 *  pObj            LPOBJECT identifying object whose string we're building.
 *                  We use the ATOMs from this object to create the string.
 *
 * Return Value:
 *  WORD            Number of characters in the string.
 */

WORD FAR PASCAL CchLinkStringCreate(LPSTR psz, LPDOCUMENT pDoc, LPOBJECT pObj)
    {
    HDC         hDC;
    WORD        cch;
    LPSTR       pszFile;
    char        szTemp[CCHPATHMAX];

    /*
     * The pszLink data is in ObjectLink format, so it immediately points
     * to a classname.  We need to translate this into a real object
     * name through the registration database.  This function in REGISTER.C
     * is simple.
     */
    GetAtomName(pObj->aClass, szTemp, CCHPATHMAX);
    cch=WDescriptionFromClass(szTemp, psz, CCHPATHMAX);

    if (0==cch)
        return 0;


    hDC=GetDC(pDoc->hList);

    //Limit the text and append a tab.
    CchTextLimit(psz, hDC, pDoc->cxList);
    lstrcat(psz, "\t");

    //Get just the filename out of the link info, limit it, and append it.
    GetAtomName(pObj->aLink, szTemp, CCHPATHMAX);
    pszFile=PszFileFromPath(szTemp);
    CchTextLimit(pszFile, hDC, pDoc->cxList);
    lstrcat(psz, pszFile);
    lstrcat(psz, "\t");


    //Append the last piece of data and a tab.
    GetAtomName(pObj->aSel, szTemp, CCHPATHMAX);
    CchTextLimit(szTemp, hDC, pDoc->cxList);
    lstrcat(psz, szTemp);
    lstrcat(psz, "\t");


    switch (pObj->dwLink)
        {
        case oleupdate_always:
            pszFile=PSZOLE(IDS_AUTOMATIC);
            break;

        case oleupdate_oncall:
            pszFile=PSZOLE(IDS_MANUAL);
            break;

        case OLEUPDATE_STATIC:
            pszFile=PSZOLE(IDS_STATIC);
            break;

        case OLEUPDATE_UNAVAILABLE:
            pszFile=PSZOLE(IDS_UNAVAILABLE);
            break;
        }
    lstrcpy(szTemp, pszFile);
    CchTextLimit(szTemp, hDC, pDoc->cxList);
    lstrcat(psz, pszFile);


    ReleaseDC(pDoc->hList, hDC);
    return lstrlen(psz);
    }





/*
 * CchTextLimit
 *
 * Purpose:
 *  Determines the number of character in psz that will fit in cx
 *  pixels in hDC and null-terminates the string at that point.
 *
 * Parameters:
 *  psz             LPSTR original string.
 *  hDC             HDC of concern, used for GetTextExtent
 *  cx              WORD number of pixels to which we limit text.
 *
 * Return Value:
 *  WORD            Number of characters in pszDst.
 */

WORD FAR PASCAL CchTextLimit(LPSTR psz, HDC hDC, WORD cx)
    {
    WORD            cch;
    WORD            cchLow;
    WORD            cchHigh;
    DWORD           dwExt;


    cch=lstrlen(psz);
    cchHigh=cch;
    cchLow=0;
    dwExt=0L;

    while (cch!=cchLow && LOWORD(dwExt)!=cx)
        {
        dwExt=GetTextExtent(hDC, psz, cch);

        if (LOWORD(dwExt)==cx)
            break;

        if (LOWORD(dwExt) > cx)
            cchHigh=cch;

        if (LOWORD(dwExt) < cx)
            cchLow=cch;

        cch=(cchHigh+cchLow) >> 1;
        }

    *(psz+cch)=0;
    return cch;
    }






/*
 * LinkStringChange
 *
 * Purpose:
 *  Changes a string in a listbox item to a new string, preserving
 *  the positioning, selection, and item data of the old string.
 *
 * Parameters:
 *  hList           HWND of the listbox.
 *  i               WORD index to the item to change.
 *  psz             LPSTR to the new string.
 *
 * Return Value:
 *  None
 */

void FAR PASCAL LinkStringChange(HWND hList, WORD i, LPSTR psz)
    {
    DWORD       dw;
    DWORD       dwSel;

    dw   =SendMessage(hList, LB_GETITEMDATA,  i, 0L);
    dwSel=SendMessage(hList, LB_GETSEL, i, 0L);

    SendMessage(hList, LB_DELETESTRING, i, 0L);
    SendMessage(hList, LB_INSERTSTRING, i, (LONG)psz);
    SendMessage(hList, LB_SETITEMDATA,  i, dw);

    if (0L!=dwSel)
        SendMessage(hList, LB_SETSEL, TRUE, MAKELONG(i, 0));

    return;
    }
