/*
 * FILEIO.C
 *
 * Functions for reading and writing .PTN files for this application.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 *
 */

#include <windows.h>
#include <ole.h>
#include "oclient.h"
#include "blackbox.h"
#include "patron.h"



/*
 * FPtnFileRead
 *
 * Purpose:
 *  Reads a .PTN file into a PATRON data structure.
 *
 * Parameters:
 *  pszFile         LPSTR of the filename to read.
 *  lpptn           LPPATRON to the structure to fill.
 *  hWndParent      HWND of the parent window of all BlackBoxes
 *  pDoc            LPDOCUMENT that owns the objects to create.
 *
 * Return Value:
 *  BOOL            TRUE if the file was successfully read,
 *                  FALSE otherwise.
 */

BOOL FAR PASCAL FPtnFileRead(LPSTR pszFile, LPPATRON lpptn,
                             HWND hWndParent, LPDOCUMENT pDoc)
    {
    HANDLE          hFile;
    OFSTRUCT        of;
    WORD            cbW;
    DWORD           cObjects;
    FILEOBJECT      fo;
    LPOBJECT        pObj;
    BOOL            fTemp;
    HWND            hWnd;
    OLESTATUS       os;


    hFile=OpenFile(pszFile, &of, OF_READ);

    if (-1==hFile)
        return FALSE;

    cbW=_lread(hFile, (LPVOID)lpptn, CBPATRON);

    if (CBPATRON!=cbW)
        {
        _lclose(hFile);
        return FALSE;
        }

    /*
     * Put the file handle in the application's OLESTREAM structure
     * so the Put method can see it.
     */
    pDoc->pStream->hFile=hFile;

    //Count the number of links that may need later updating.
    pDoc->cLinks=0;

    //Read each object and create a window if necessary.
    cObjects=lpptn->cObjects;

    while (cObjects)
        {
        cObjects--;

        //Read the FILEOBJECT structure for this object
        cbW=_lread(hFile, (LPSTR)&fo, CBFILEOBJECT);

        if (CBFILEOBJECT!=cbW)
            break;

        pObj=PObjectAllocate(&fTemp, pDoc);

        if (!fTemp)
            {
            PObjectFree(pDoc, pObj);
            break;
            }

        //Attempt to create an OLEOBJECT for this junk in the file.
        os=OleLoadFromStream((LPOLESTREAM)pDoc->pStream, PSZOLE(IDS_STDFILEEDITING),
                             (LPOLECLIENT)pObj, pDoc->lh, fo.szName, &pObj->pObj);

        /*
         * Since we need to call other OLE functions for this object when
         * we create a BlackBox window, we need to wait for each one
         * individually.
         */
        if (OLE_OK!=OsError(os, pDoc, pObj, TRUE))
            {
            PObjectFree(pDoc, pObj);
            break;
            }

        //Create a window for this object with the stored information
        hWnd=HBlackBoxCreate(hWndParent, &fo, TRUE, pObj);

        if (NULL==hWnd)
            break;

        /*
         * Try to update an automatic link.  If we can't then count another
         * link to update later.
         */
        if (OT_LINK==pObj->dwType)
            {
            //Count a manual or non-updated auto link
            if (!FObjectAutoLinkUpdate(pDoc, pObj))
                pDoc->cLinks++;
            }
        }

    _lclose(hFile);

    //Activate the first window we created.
    hWnd=GetWindow(hWndParent, GW_CHILD);

    if (NULL!=hWnd)
        {
        BringWindowToTop(hWnd);
        SendMessage(hWnd, WM_NCACTIVATE, TRUE, 0L);
        }

    return TRUE;
    }





/*
 * FPtnFileWrite
 *
 * Purpose:
 *  Writes a .PTN file in a PATRON structure pointed to by lpptn.
 *
 * Parameters:
 *  pszFile         LPSTR of the filename to read.
 *  lpptn           LPPATRON pointing to the structure to write,
 *  pDoc            LPDOCUMENT that owns the objects.
 *
 * Return Value:
 *  BOOL            TRUE if the file was successfully written,
 *                  FALSE otherwise.
 */

BOOL FAR PASCAL FPtnFileWrite(LPSTR pszFile, LPPATRON lpptn, LPDOCUMENT pDoc)
    {
    HANDLE          hFile;
    OFSTRUCT        of;
    DWORD           cb;
    BOOL            fOK;

    hFile=OpenFile(pszFile, &of, OF_CREATE | OF_WRITE);

    if (-1==hFile)
        return FALSE;

    cb=_lwrite(hFile, (LPSTR)lpptn, CBPATRON);

    if ((DWORD)CBPATRON!=cb)
        {
        _lclose(hFile);
        return FALSE;
        }

    /*
     * Put the file handle in the application's OLESTREAM structure
     * so the Put method can see it.
     */
    pDoc->pStream->hFile=hFile;

    //Go write all the objects.
    fOK=FObjectsEnumerate(pDoc, FEnumFileWrite, 0L);
    _lclose(hFile);


    /*
     * Leave the call to OleSavedClientDoc to the caller of this function.
     * We don't know anything about documents here, just objects.
     */
    return fOK;
    }




/*
 * FEnumFileWrite
 *
 * Purpose:
 *  Enumeration callback function for use from FPtnFileWrite when it
 *  calls FObjectsEnumerate.  For each object we save the FILEOBJECT
 *  header for the window and call OleSaveToStream for it's object.
 *
 * Parameters:
 *  pDoc            LPDOCUMENT identifying the owner of all objects.
 *  pObj            LPOBJECT identifying the current object.
 *  dw              DWORD for extra data, unused.
 *
 * Return Value:
 *  BOOL            TRUE to continue the enumeration, FALSE otherwise.
 */

BOOL FAR PASCAL FEnumFileWrite(LPDOCUMENT pDoc, LPOBJECT pObj, DWORD dw)
    {
    FILEOBJECT      fo;
    OLESTATUS       os;
    HWND            hWnd;
    WORD            cb;

    hWnd=(HWND)pObj->hData;

    //Retrieve the window rectangle for this blackbox and write it.
    SendMessage(hWnd, BBM_RECTGET, 0, (LONG)(LPSTR)&fo.rc);
    GetWindowText(hWnd, fo.szName, 40);
    fo.wID=GetWindowWord(hWnd, GWW_ID);

    /*
     * Saving the size of an object is useful to skip over objects
     * on a quick file scan.  For example, if you want to quickly open
     * a file without showing pictures, you could read the FILEOBJECT
     * header and seek past the OLE object.  You could then go back
     * later and call OleLoadFromStream.
     *
     * Here, if we cannot get the size, we fail the file write.
     */
    os=OleQuerySize(pObj->pObj, &fo.cbObject);

    if (OLE_OK!=os)
        return FALSE;

    //Write the object header.
    cb=_lwrite(pDoc->pStream->hFile, (LPSTR)&fo, CBFILEOBJECT);

    if (CBFILEOBJECT!=cb)
        return FALSE;

    //Write the actual OLE object.
    os=OleSaveToStream(pObj->pObj, (LPOLESTREAM)pDoc->pStream);

    //Stop enumeration if there's an error.
    return (OLE_OK==os);
    }
