/*
 * CLIP.C
 *
 * Functions to interact with the clipboard and one to enable or disable
 * menu items depending on the state of the clipboard.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 *
 */

#include <windows.h>
#include <ole.h>
#include "oclient.h"
#include "blackbox.h"
#include "patron.h"




/*
 * MenuClipboardEnable
 *
 * Purpose:
 *  Enabled or disables the Edit menu command of Cut, Copy, and Paste.
 *  the OLE formats are handled through a call to MenuOLEClipboardEnable,
 *  which occurs first to allow this function to enable Paste if there
 *  still is a non-OLE format we can paste.  Note, however, that in this
 *  application we do not paste anything but OLE objects.
 *
 *  Call this function from the the WM_INITPOPUPMENU message case for the
 *  main window.
 *
 * Parameters:
 *  hMenu           HMENU of the Edit popup menu.
 *  pGlob           LPGLOBALS to the global variable block.
 *
 * Return Value:
 *  None
 */

void FAR PASCAL MenuClipboardEnable(HMENU hMenu, LPGLOBALS pGlob)
    {
    WORD        wTemp;

    //Cannot cut, copy, or clear without an active window.
    wTemp=(NULL!=pGlob->hWndLastActive) ? MF_ENABLED : (MF_DISABLED | MF_GRAYED);
    EnableMenuItem(hMenu, IDM_EDITCOPY,            wTemp | MF_BYCOMMAND);
    EnableMenuItem(hMenu, IDM_EDITCUT,             wTemp | MF_BYCOMMAND);
    EnableMenuItem(hMenu, IDM_EDITCONVERTTOSTATIC, wTemp | MF_BYCOMMAND);
    EnableMenuItem(hMenu, IDM_EDITCLEAR,           wTemp | MF_BYCOMMAND);
    EnableMenuItem(hMenu, IDM_EDITCLEARALL,        wTemp | MF_BYCOMMAND);

    /*
     * If we pasted anything else, then we would check for those formats
     * here and enable the Paste menu item if they were available.  Since
     * we don't, then we don't mess with the Paste item.
     */
    return;
    }





/*
 * FEditCut
 *
 * Purpose:
 *  Places a copy of an object in the currently active window on the
 *  clipboard then deletes that object and its window.
 *
 * Parameters:
 *  pGlob           LPGLOBALS to the global variable block.
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 *
 */

BOOL FAR PASCAL FEditCut(LPGLOBALS pGlob)
    {
    //Copy editor contents to the clipboard.
    if (!FEditCopy(pGlob, TRUE))
        return FALSE;

    //Delete the last active window, which is the one we copy.
    SendMessage(pGlob->hWnd, WM_COMMAND, IDM_EDITCLEAR, 0L);
    return TRUE;
    }




/*
 * FEditCopy
 *
 * Purpose:
 *  Places a copy of an object in the currently active window on the
 *  clipboard.
 *
 * Parameters:
 *  pGlob           LPGLOBALS to the global variable block.
 *  fCut            BOOL indiciating if we are being used from FEditCut.
 *
 * Return Value:
 *  BOOL            TRUE if anything was successfully placed on the
 *                  clipboard, FALSE otherwise.
 *
 */

BOOL FAR PASCAL FEditCopy(LPGLOBALS pGlob, BOOL fCut)
    {
    LPOBJECT            pObj;
    OLESTATUS           os;

    //Attempt to gain clipboard ownership.
    if (!OpenClipboard(pGlob->hWnd))
        return FALSE;

    //Copying an object to the clipboard could not be easier.
    pObj=(LPOBJECT)SendMessage(pGlob->hWndLastActive, BBM_POBJECTGET, 0, 0L);
    os=OleCopyToClipboard(pObj->pObj);

    //Free clipboard ownership.
    CloseClipboard();

    return (OLE_OK==os);
    }




/*
 * FEditPaste
 *
 * Purpose:
 *  Retrieves an embedded or linked object from the clipboard and sets
 *  it as the OLE object for a newly created window.
 *
 *  Note that if this function is called, then the clipboard format
 *  is available because the Paste menu item is only enabled if the
 *  format is present.
 *
 * Parameters:
 *  pGlob           LPGLOBALS to the global variable block.
 *  fLink           BOOL indicates if we want to Paste a linked object
 *                  instead of an embedded object.
 *  pDoc            LPDOCUMENT containing OLE information.
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 *
 */

BOOL FAR PASCAL FEditPaste(LPGLOBALS pGlob, BOOL fLink, LPDOCUMENT pDoc)
    {
    HWND            hWndT;
    FILEOBJECT      fo;
    OLESTATUS       os;
    LPOBJECT        pObj;
    BOOL            fTemp;


    if (!OpenClipboard(pGlob->hWnd))
        return FALSE;


    FileObjectDefaults(&fo);
    pObj=PObjectAllocate(&fTemp, pDoc);

    if (!fTemp)
        {
        PObjectFree(pDoc, pObj);
        CloseClipboard();
        return FALSE;
        }


    if (fLink)
        {
        //Attempt to create a linked object.
        os=OleCreateLinkFromClip(PSZOLE(IDS_STDFILEEDITING), (LPOLECLIENT)pObj,
                                 pDoc->lh, fo.szName, &pObj->pObj, olerender_draw, 0);

        os=OsError(os, pDoc, pObj, TRUE);

        //On failure we'll try to create an embedded object.
        if (OLE_OK!=os)
            fLink=FALSE;
        }

    if (!fLink)
        {
        //Attempt to create the embedded object.
        os=OleCreateFromClip(PSZOLE(IDS_STDFILEEDITING), (LPOLECLIENT)pObj,
                             pDoc->lh, fo.szName, &pObj->pObj, olerender_draw, 0);

        //If an embedded object fails, try static.
        os=OsError(os, pDoc, pObj, TRUE);

        if (OLE_OK!=os)
            {
            os=OleCreateFromClip(PSZOLE(IDS_STATIC), (LPOLECLIENT)pObj,
                                 pDoc->lh, fo.szName, &pObj->pObj, olerender_draw, 0);

            os=OsError(os, pDoc, pObj, TRUE);
            }
        }


    CloseClipboard();

    /*
     * Generally OleCreateFromClip will not return OLE_WAIT_FOR_RELEASE
     * but we need to be prepared in case it does.
     */
    if (OLE_OK!=os)
        {
        PObjectFree(pDoc, pObj);
        return FALSE;
        }

    //This frees pObj on failure.
    hWndT=HBlackBoxCreate(pGlob->hWnd, &fo, TRUE, pObj);

    if (NULL==hWndT)
        return FALSE;

    //Update the size and image in the window.
    SendMessage(hWndT, BBM_OBJECTNOTIFY, OLE_CHANGED, (LONG)pObj);
    return TRUE;
    }





/*
 * FEditConvertToStatic
 *
 * Purpose:
 *  Converts any object into a static object that contains the same
 *  image but cannot be activated in any way.  The data is still held in
 *  OLECLI but the object is frozen.
 *
 * Parameters:
 *  pGlob           LPGLOBALS to the global variable block.
 *  pDoc            LPDOCUMENT containing OLE information.
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 */

BOOL FAR PASCAL FEditConvertToStatic(LPGLOBALS pGlob, LPDOCUMENT pDoc)
    {
    LPOLEOBJECT     pOLEObj;
    LPOLEOBJECT     pOLEObjT;
    LPOBJECT        pObj;
    OLESTATUS       os;
    char            szObject[128];
    char            szName[128];


    pObj=(LPOBJECT)SendMessage(pGlob->hWndLastActive, BBM_POBJECTGET, 0, 0L);

    GetAtomName(pObj->aName, szName, 128);

    //Create a temporary name for the converted object.
    lstrcpy(szObject, PSZOLE(IDS_STATIC));

    os=OleObjectConvert(pObj->pObj, PSZOLE(IDS_STATIC), (LPOLECLIENT)pObj,
                        pDoc->lh, szName, &pOLEObj);

    if (OLE_OK!=os)
        return TRUE;

    //Delete the old object and wait on it.
    os=OleDelete(pObj->pObj);

    if (OLE_OK!=OsError(os, pDoc, pObj, TRUE))
        {
        //Could not convert, so delete the new object.
        os=OleDelete(pOLEObj);

        //OsError uses pObj->pObj for waiting.
        pOLEObjT=pObj->pObj;
        pObj->pObj=pOLEObj;

        OsError(os, pDoc, pObj, TRUE);

        pObj->pObj=pOLEObjT;
        return FALSE;
        }

    pObj->pObj=pOLEObj;
    OleRename(pObj->pObj, szName);

    //Insure that we're marked as static.
    pObj->dwType = OT_STATIC;
    return TRUE;
    }
