## Copyright (C) 1996 John W. Eaton
##
## This file is part of Octave.
##
## Octave is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
##
## Octave is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with Octave; see the file COPYING.  If not, write to the Free
## Software Foundation, 59 Temple Place - Suite 330, Boston, MA
## 02111-1307, USA.

## Save a matrix to disk in image format.
##
## saveimage (filename, x) saves matrix x to file filename in octave's
## image format.  The current colormap is saved in the file also.
##
## saveimage (filename, x, "img") saves the image in the default format
## and is the same as saveimage (filename, x).
##
## saveimage (filename, x, "ppm") saves the image in ppm format instead
## of the default octave image format.
##
## saveimage (filename, x, "ps") saves the image in PostScript format
## instead of the default octave image format. (Note: images saved in
## PostScript format can not be read back into octave with loadimage.)
##
## saveimage (filename, x, format, map) saves the image along with the
## specified colormap in the specified format.
##
## Note: If the colormap contains only two entries and these entries
## are black and white, the bitmap ppm and PostScript formats are used.
## If the image is a gray scale image (the entries within each row of
## the colormap are equal) the gray scale ppm and PostScript image
## formats are used, otherwise the full color formats are used.
##
## The conversion to PostScript is based on pbmtolps.c, which was
## written by
##
##   George Phillips <phillips@cs.ubc.ca>
##   Department of Computer Science
##   University of British Columbia
##
## and is part of the portable bitmap utilities,
##
## SEE ALSO: loadimage, save, load, colormap

## Author: Tony Richardson <amr@mpl.ucsd.edu>
## Created: July 1994
## Adapted-By: jwe

## Modified by Klaus Gebhardt, 1996

function saveimage (filename, img, img_form, map)

  if (nargin < 2 || nargin > 4)
    usage ("saveimage (filename, matrix, [format, [colormap]])");
  endif

  if (nargin < 4)
    map = colormap ();
  endif

  [map_nr, map_nc] = size (map);

  if (map_nc != 3)
    error ("colormap should be an N x 3 matrix");
  endif

  if (nargin < 3)
    img_form = "img";
  elseif (! isstr (img_form))
    error ("image format specification must be a string");
  elseif (! (strcmp (img_form, "img")
             || strcmp (img_form, "ppm")
             || strcmp (img_form, "gif")
             || strcmp (img_form, "bmp")
	     || strcmp (img_form, "ps")))
    error ("unsupported image format specification");
  endif

  if (! is_matrix (img))
    warning ("image variable is not a matrix");
  endif

  if (! isstr (filename))
    error ("file name must be a string");
  endif

  ## If we just want Octave image format, save and return.

  if (strcmp (img_form, "img"))
    eval (strcat ("save -ascii ", filename, " map img"));
    return;
  endif

  ## Convert to another format if requested.

  grey = all (map(:,1) == map(:,2) && map(:,1) == map (:,3));

  map_sz = map_nr * map_nc;

  map = reshape (map, map_sz, 1);

  idx = find (map > 1);
  map (idx) = ones (size (idx));

  idx = find (map < 0);
  map (idx) = zeros (size (idx));

  map = round (255 * map);

  bw = (map_nr == 2
        && ((map(1,1) == 0 && map(2,1) == 255)
            || (map(1,1) == 255 && map(2,1) == 0)));

  img = round (img');
  [img_nr, img_nc] = size (img);

  img_sz = img_nr * img_nc;
  img = reshape (img, img_sz, 1);

  idx = find (img >= map_nr - 1);
  img (idx) = ones (size (idx)) * map_nr;

  idx = find (img <= 0);
  img (idx) = ones (size (idx));

  if (!strcmp (img_form, "ppm"))
    temp_file = octave_tmp_file_name;
  else
    temp_file = filename;
  endif

  if (grey && map_nr == 2 && bw)

    map = map(:,1);

    if (map(1) != 0)
      map(1) = 1;
    else
      map(2) = 1;
    endif

    img = map(img);
    n_long = rem (img_sz, 8);
    if (n_long == 0)
      n_long = 8;
    else
      n_long = 1 + nlong;
    endif

    idx = 1:8:img_sz;
    s_len = length (idx) - 1;

    tmp = img (1:8:img_sz) * 128;
    for i = 2:n_long
      tmp = tmp + img (i:8:img_sz) * 2^(8-i);
    endfor
    for i = (n_long+1):8
      tmp(1:s_len) = tmp(1:s_len) + img (i:8:img_sz) * 2^(8-i);
    endfor

    fid = fopen (temp_file, "w");
    fprintf (fid, "P4\n%d %d\n", img_nr, img_nc);
    fwrite (fid, tmp, "char");
    fprintf (fid, "\n");
    fclose (fid);

  elseif (grey)

    fid = fopen (temp_file, "w");
    fprintf (fid, "P5\n%d %d\n255\n", img_nr, img_nc);
    fwrite (fid, map(img), "uchar");
    fprintf (fid, "\n");
    fclose (fid);

  else

    img_idx = ((1:3:3*img_sz)+2)';
    map_idx = ((2*map_nr+1):map_sz)';

    tmap = map(map_idx);
    tmp(img_idx--) = tmap(img);

    map_idx = map_idx - map_nr;
    tmap = map(map_idx);
    tmp(img_idx--) = tmap(img);

    map_idx = map_idx - map_nr;
    tmap = map(map_idx);
    tmp(img_idx--) = tmap(img);

    fid = fopen (temp_file, "w");
    fprintf (fid, "P6\n%d %d\n255\n", img_nr, img_nc);
    fwrite (fid, tmp, "uchar");
    fprintf (fid, "\n");
    fclose (fid);

  endif

  if (strcmp (img_form, "ps") == 1)
    cmd = sprintf ("pnmtops %s > %s & rm -f %s", temp_file,filename,temp_file);
  elseif (strcmp (img_form, "bmp") == 1)
    cmd = sprintf ("ppmtobmp %s > %s & rm -f %s",temp_file,filename,temp_file);
  elseif (strcmp (img_form, "gif") == 1)
    cmd = sprintf ("ppmtogif %s > %s & rm -f %s",temp_file,filename,temp_file);
  endif
  
  if (!strcmp (img_form, "ppm") == 1)
    system (cmd);
  endif

endfunction
