/*
 * File......: CRQJBFI.PRG
 * Author....: Glenn Scott and Chris Boduch
 * CIS ID....: 71620,1521
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Glenn Scott and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

/*  $DOC$
 *  $FUNCNAME$
 *     Overview
 *  $CATEGORY$
 *     Queue
 *  $ONELINER$
 *     Design Philosphy of Clipper APIs to access QMS
 *  $SYNTAX$
 *
 *  $ARGUMENTS$
 *
 *  $RETURNS$
 *
 *  $DESCRIPTION$
 *
 *     The goal of this Clipper API is to provide as much access
 *     to the QMS (Queue Management System) APIs as possible,
 *     while trying to make all the data available in a Clipper-esque
 *     way.  For example, where Novell says a date is a 3 byte string
 *     for month, day, and year, we want to use a Clipper date type
 *     instead.
 *
 *     Queue jobs reside in queues, and every queue job has a set
 *     of attributes which Novell calls the Job Entry Structure.
 *     These attributes include numeric things like "job number"
 *     and "target server ID" as well as character things like
 *     "Job Description" and "Client Record Area."
 *
 *     We have therefore created an entity called a JOBSTRUCT which
 *     is a Clipper array.  Right now this array has 23 elements.
 *     Each of the elements has a #define associated with it in
 *     netto.CH, with the prefix QJ (queue job, I guess).
 *
 *     Therefore, we know that in any JOBSTRUCT aJob, aJob[ QJ_CLIENT_ID ]
 *     gives you the Client ID, etc.
 *
 *     The JOBSTRUCT is not in a format the Novell NetWare will ever
 *     understand.  It is what I call "fully translated", meaning, you
 *     and I and Clipper can make sense of it.
 *
 *     In order to create a jobstruct, I have defined a command in
 *     netto.CH called CREATE JOBSTRUCT.   Using this, you can very,
 *     very easily create the JOBSTRUCT array.
 *
 *     Once you have our JOBSTRUCT, you can pass it to any routine
 *     that requires a JOBSTRUCT, such as fn_crqJbFi() (Create Queue
 *     Job and File).
 *
 *     Those routines will call some "internals" that convert the
 *     array to a big long character string that NetWare wants to
 *     see.
 *
 *     If a routine _returns_ a JOBSTRUCT, then the reverse is
 *     happening:  NetWare returns a big long character string that
 *     is parsed and translated into a JOBSTRUCT we all prefer.
 *
 *     <describe the parts of the JOBSTRUCT>
 *
 *     <describe QMS in general?>
 *
 *     <rough Queue Q&As>
 *
 *      Q:  How do I get a list of print queues on all servers?
 *      A:  First, you use fn_pfConID() to save the current preferred
 *          connection ID (server connection).  Then, you fn_sPfCID()
 *          to set the preferred connection ID to one of the other
 *          connections (1-8).
 *
 *          For each of these other connections, call Scan Bindery
 *          Object [ fn_scaBndO() ] searching for print queues
 *          (object type OT_PRINT_QUEUE).  When you're all done,
 *          reset the preferred connection ID back to the original.
 *
 *          Better yet, write this as a high level function and submit
 *          it for the next release of this library.
 *
 *  $EXAMPLES$
 *
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */


/*  $DOC$
 *  $FUNCNAME$
 *     FN_CRQJBFI()
 *  $CATEGORY$
 *     Queue
 *  $ONELINER$
 *     Create queue job and file
 *  $SYNTAX$
 *
 *     fn_crQJbFi( nQueueID, @aJobStruct, [<lSetPrt>] ) -> nHandle
 *
 *  $ARGUMENTS$
 *
 *    <nQueueID> is the bindery object ID for the queue.
 *    <aJobStruct> is an array containing a queue "job structure."
 *    MUST BE PASSED BY REFERENCE!
 *
 *    The user-defined command CREATE JOBSTRUCT can help you create
 *    a properly formatted job structure.
 *
 *    <lSetPrt> is an optional logical argument.  If you set it
 *    to .t., then the SET PRINTER DEVICE will be redirected 
 *    such that your printer output will go directly to the spool
 *    file created by this function.  Defaults to FALSE (.f.).
 *
 *  $RETURNS$
 *
 *    <nHandle>
 *
 *    Also, the <aJobStruct> will be filled in...
 *
 *  $DESCRIPTION$
 *
 *    It is important that if you set <lSetPrt> to .t., that you 
 *    let the fn_cfabjq() (Close File and Abort Queue Job) or
 *    fn_stqjob() (Close File and Start Queue Job) functions 
 *    close the SET PRINTER device -- in other words, don't use
 *    a SET PRINTER TO all by itself.
 *
 *  $EXAMPLES$
 *
 *        #include "netto.ch"
 *        // Get the queue id for the queue, and set up the job
 *        // structure, then...
 *                     .
 *                     .
 *        hJobFile := fn_crqjbfi( nQid, @aJob, .t. )
 *        set device to print
 *        set printer on
 *
 *        ? "Hello there, this is printed from Clipper!"
 *        ? "Is it not wonderful?"
 *        eject
 *        ? "Next page!"
 *        @ 20, 10 say "And this is at 20, 10!"
 *
 *        set printer off
 *        set device to screen
 *
 *        if !fn_stqjob( nQid, aJob[ QJ_JOB_NUMBER ], hJobFile )
 *           qout( "Error closing queue job:", fn_error() )
 *        endif
 *
 *  $SEEALSO$
 *      CREATE JOBSTRUCT
 *  $INCLUDE$
 *      netto.CH
 *  $END$
 */

#include "netto.ch"

function fn_crQJbFi( nQueueID, aJobStruct, lSetPrt )
   local cReq, cRep, lResult := .f., cJob, nHandle := -1, xSavErr

   default lSetPrt to .f.

   if pcount() >= 2
      cJob := _fnJs2pkt( aJobStruct )

      cReq := I2BYTE( 104 ) + ;            // 68h
              L2HILO( nQueueID ) + ;
              cJob

      cRep := repl( chr(0), 54 )

      if _fnReq( 227, cReq, @cRep ) == ESUCCESS
         cJob       := cRep + subs( cJob, 55 )
         aJobStruct := _fnPkt2Js( cJob )
         lResult    := .t.
      endif

   else
      _fnSetErr( EBADPARM )
   endif

   // NetWare 4.0 note: fopen()ing NETQ may not work under the 
   // Requester; at least it didn't in the betas and Novell 
   // said they were working on it.

   if lResult
      nHandle := fopen( "NETQ", 2 )    // Read-Write
      if nHandle # -1 .and. lSetPrt
         xSavErr := fn_error()
         fn_setprc( nHandle )
         _fnSetErr( xSavErr )
      endif
   endif

   return nHandle
