/*
 * File......: FSNAME.PRG
 * Author....: Steve Larsen
 * CIS ID....: 76370,1532
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Steve Larsen and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

#include "ftint86.ch"
#include "netto.ch"

/*  $DOC$
 *  $FUNCNAME$
 *      fn_FSName()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      GetFileServerNameTable
 *  $SYNTAX$
 *
 *      fn_FSName() -> aNameTable
 *
 *  $ARGUMENTS$
 *
 *      None
 *
 *  $RETURNS$
 *
 *      An 8 element array containing the names of each file server that
 *      the calling workstation is currently attached to.
 *
 *  $DESCRIPTION$
 *
 *      This function returns a copy of the shell's File Server Name Table.
 *
 *      A workstation may be attached to up to 8 file servers.  For each
 *      that the workstation is attached, an entry is made in this table and
 *      the ConnectionID table.  The index into the table is the server's
 *      ConnectionID.
 *
 *      Server names can be up to 47 characters in length.  If no server
 *      is active at a particular ID, the Name Table will contain an empty
 *      string "".
 *
 *  $EXAMPLES$
 *
 *      /* List the servers that this station is attached to */
 *
 *      aServers := fn_FSName()
 *
 *      AEVAL( aServers, {|e| QOUT( e ) } )
 *
 *  $SEEALSO$
 *      fn_connID()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_FSName()
	LOCAL aRegs[ INT86_MAX_REGS ], cReply := '', aReply[ 8 ], nSeg, nOff
	LOCAL i

	aRegs[ AX ] := ft_HEX2DEC( "EF04" )   // Netware Service EFh, Function 04

	IF !ft_int86( INT21, aRegs )          // fetch address of FSName table
		_fnSetErr( EINT86 )
 	ELSE
		nSeg := aRegs[ ES ]
		nOff := aRegs[ SI ]
		FOR i := 1 TO 8                    // fill array with Server Names
  			aReply[ i ] := fn_NoNull( fn_PeekStr( nSeg, @nOff, 48 ) ) // read a table
		NEXT
	ENDIF

RETURN( aReply )


/*  $DOC$
 *  $FUNCNAME$
 *      fn_wrFSName()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      WriteFileServerNameTable
 *  $SYNTAX$
 *
 *      fn_wrFSName( <aNameTable> ) -> NIL
 *
 *  $ARGUMENTS$
 *
 *      <aNameTable> is an 8 element array of file server names,
 *      corresponding to entries in the Connection ID table.
 *
 *      File server names may be up to 47 characters in length.
 *
 *  $RETURNS$
 *
 *      Nothing
 *
 *  $DESCRIPTION$
 *
 *      Use this function when making changes to the Connection ID table.  For
 *      example, the function fn_attchFS() writes the name of the target file
 *      server, as well as connection ID info, before attempting to attach to
 *      a new file server.
 *
 *  $EXAMPLES$
 *
 *      /* change a file server's local name to something more meaningful */
 *
 *      aFSNames := fn_FSName()
 *
 *      IF ( nID := ASCAN( aTable, {|e| e == "FS3" } ) != 0 )
 *
 *         aTable[ nID ] := "Server Next to Sally's Desk"
 *         fn_wrFSName( aTable )
 *
 *      ENDIF
 *
 *
 *      /* NOTE: Although the above example serves to demonstrate how to
 *         use fn_wrFSName(), the practice illustrated above is not recom-
 *         mended.  fn_wrFSName() only writes to the Shell's file server
 *         name table, not to the network.  In the example above, the file
 *         server "Server Next to Sally's Desk" would still have to be
 *         referenced as "FS3" by any other file server functions, such as
 *         attaching or logging in. */
 *
 *  $SEEALSO$
 *      fn_FSName() fn_connID() fn_attchFS()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_wrFSName( aTable )
	LOCAL aRegs[ INT86_MAX_REGS ], i, nSeg, nOff

	aRegs[ AX ] := ft_HEX2DEC( "EF04" )   // Netware Service EFh, Function 04

	IF !ft_int86( INT21, aRegs )          // fetch address of FSName table
		_fnSetErr( EINT86 )
  	ELSE

		nSeg := aRegs[ ES ]
		nOff := aRegs[ SI ]

		FOR i := 1 TO 8                    // write Server Names to memory

			fn_PokeStr( nSeg, @nOff, PADR( aTable[ i ], 48, I2BYTE(0) ) )

		NEXT

	ENDIF

RETURN( NIL )
