/*
 * File......: CONNID.PRG
 * Author....: Steve Larsen
 * CIS ID....: 76370,1532
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Steve Larsen and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

#include "ftint86.ch"
#include "netto.ch"

#define    IDGET    1
#define    IDSET    2

/*  These are important, don't change the numbers:  */

#define    GET_DEFAULT    2
#define    GET_PREFERRED  1
#define    SET_PREFERRED  0
#define    GET_PRIMARY    5
#define    SET_PRIMARY    4


/*  $DOC$
 *  $FUNCNAME$
 *      fn_connID()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      GetConnectionID
 *  $SYNTAX$
 *
 *      fn_connID() -> aConnectionTable
 *
 *  $ARGUMENTS$
 *
 *      None
 *
 *  $RETURNS$
 *
 *      The shell's Connection ID Table is an 8 element array.
 *      Each element of the array consists of a 12 element subarray,
 *      filled with info from the Connection ID Table (aka File
 *      Server Table), described below.
 *
 *      All entries in the array are numeric decimal values, with
 *      the exception of the server address.
 *
 *  $DESCRIPTION$
 *
 *      Each workstation shell has a Connection ID table and a
 *      File Server Name Table.  The Connection ID Table consists
 *      of up to eight entries (1 to 8) that are each 32 bytes
 *      in length.  Each of the eight entries in the Connection ID
 *      Table contains the following information:
 *
 *      Slot in Use (0=empty, FFh=used)                INTEGER
 *      Server Order Number (1 to 8)                   INTEGER
 *      Server's Network Number                        cHexStr[ 8]
 *      Physical Node Address                          cHexStr[12]
 *      Socket Number                                  cHexStr[ 4]
 *      Receive Timeout                                INTEGER
 *      Router's physical node address                 INTEGER
 *      Packet Sequence Number                         INTEGER
 *      Connection Number   (FFh=no connection)        INTEGER
 *      Connection Status (0=connection functioning)   INTEGER
 *      Maximum Timeout                                INTEGER
 *      Filler (Reserved)                              INTEGER
 *
 *  $EXAMPLES$
 *
 *      /* list connection ID table for each attached server */
 *
 *      aTbl   := fn_connID()    // GetConnectionID Table
 *      aFSNameTable := fn_FSName()    // GetFileServerName Table
 *
 *      FOR i := 1 TO 8                // up to 8 attached servers
 *        IF aTbl[ i, 1 ] == 0   // 0=empty, 255=in use
 *          LOOP
 *        ENDIF
 *        ?
 *        ?'Connection ID table for server: ' + aFSNameTable[ i ]
 *        ?
 *        ?' 1-Slot in Use (0=empty, FF=used)        :', aTbl[i, 1]
 *        ?' 2-Server Order Number (1 to 8)          :', aTbl[i, 2]
 *        ?' 3-Servers Network Number                :', aTbl[i, 3]
 *        ?' 4-Physical Node Address                 :', aTbl[i, 4]
 *        ?' 5-Socket Number                         :', aTbl[i, 5]
 *        ?' 6-Receive Timeout                       :', aTbl[i, 6]
 *        ?' 7-Routers physical node address         :', aTbl[i, 7]
 *        ?' 8-Packet Sequence Number                :', aTbl[i, 8]
 *        ?' 9-Connection Number (FFh=no conn)       :', aTbl[i, 9]
 *        ?'10-Connection Status (0=conn functioning):', aTbl[i,10]
 *        ?'11-Maximum Timeout                       :', aTbl[i,11]
 *        ?'12-Filler (Reserved)                     :', aTbl[i,12]
 *
 *        NEXT
 *
 *  $SEEALSO$
 *      fn_FSName() fn_connInf() fn_attchFS()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_connID()
   LOCAL aRegs[ INT86_MAX_REGS ], aReply[ 8 ], i, nSeg, nOff

   aRegs[ AX ] := ft_HEX2DEC( "EF03" )   // Netware Service EFh, Function 03

   IF !ft_int86( INT21, aRegs )          // fetch address of ConnID table
      _fnSetErr( EINT86 )
     ELSE

   nSeg := aRegs[ ES ]                   // store address
   nOff := aRegs[ SI ]

      FOR i := 1 TO 8                    // fill sub-arrays with tables
         aReply[ i ] := ARRAY( 12 )

         aReply[ i ][  1 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 1 ) )
         aReply[ i ][  2 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 1 ) )
         aReply[ i ][  3 ] := PADL( fn_bin2hex( fn_PeekStr( nSeg, @nOff, 4 ) ),  8, '0' )
         aReply[ i ][  4 ] := PADL( fn_bin2hex( fn_PeekStr( nSeg, @nOff, 6 ) ), 12, '0' )
         aReply[ i ][  5 ] := PADL( fn_bin2hex( fn_PeekStr( nSeg, @nOff, 2 ) ),  4, '0' )
         aReply[ i ][  6 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 2 ) )
         aReply[ i ][  7 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 6 ) )
         aReply[ i ][  8 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 1 ) )
         aReply[ i ][  9 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 1 ) )
         aReply[ i ][ 10 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 1 ) )
         aReply[ i ][ 11 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 2 ) )
         aReply[ i ][ 12 ] := fn_bin2i( fn_PeekStr( nSeg, @nOff, 5 ) )

      NEXT
   ENDIF

RETURN( aReply )


/*  $DOC$
 *  $FUNCNAME$
 *      fn_wrConnID()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      WriteConnectionIDTable
 *  $SYNTAX$
 *
 *      fn_wrConnID( <aConnectTable> ) -> NIL
 *
 *  $ARGUMENTS$
 *
 *      <aConnectTable>: An array containing the new Connection ID Table.
 *
 *  $RETURNS$
 *
 *      Nothing.
 *
 *  $DESCRIPTION$
 *
 *      Use this function to make changes to the shell's Connection ID
 *      table.  Used internally by fn_attchFS().
 *
 *  $EXAMPLES$
 *
 *      fn_wrConnID( aNewConnId )
 *
 *  $SEEALSO$
 *      fn_connID() fn_attchFS()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_wrConnID( aTable )
   LOCAL aRegs[ INT86_MAX_REGS ], nSeg, nOff, i, cTable

   aRegs[ AX ] := ft_HEX2DEC( "EF03" )   // Netware Service EFh, Function 03

   IF !ft_int86( INT21, aRegs )          // fetch address of ConnID table
      _fnSetErr( EINT86 )
     ELSE

   nSeg := aRegs[ ES ]
   nOff := aRegs[ SI ]

      FOR i := 1 TO 8                    // fill sub-arrays with tables

         cTable := fn_i2bin(   aTable[ i ][  1 ], 1 )+; // SlotInUse
                    fn_i2bin(   aTable[ i ][  2 ], 1 )+; // ServerOrderNumber
                   fn_hex2bin( aTable[ i ][  3 ], 4 )+; // ServerInternalNetNo
                   fn_hex2bin( aTable[ i ][  4 ], 6 )+; // ServerPhysNodeAddr
                   fn_hex2bin( aTable[ i ][  5 ], 2 )+; // SocketNo
                   fn_i2bin(   aTable[ i ][  6 ], 2 )+; // ReceiveTimeOut
                   fn_i2bin(   aTable[ i ][  7 ], 6 )+; // RouterPhysNodeAddr
                   fn_i2bin(   aTable[ i ][  8 ], 1 )+; // PacketSeqNo
                   fn_i2bin(   aTable[ i ][  9 ], 1 )+; // ConnectionNo
                   fn_i2bin(   aTable[ i ][ 10 ], 1 )+; // ConnectionStatus
                   fn_i2bin(   aTable[ i ][ 11 ], 2 )+; // MaxTimeOut
                   fn_i2bin(   aTable[ i ][ 12 ], 5 )     // Reserved

         /* write the table to memory */

         fn_PokeStr( nSeg, @nOff, cTable )

      NEXT
   ENDIF

RETURN( NIL )

/*  $DOC$
 *  $FUNCNAME$
 *      fn_defCID()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      GetDefaultConnectionID
 *  $SYNTAX$
 *
 *      fn_defCID() -> nConnectionID
 *
 *  $ARGUMENTS$
 *
 *      None
 *
 *  $RETURNS$
 *
 *      The connection ID for the default file server, else -1 if error.
 *
 *  $DESCRIPTION$
 *
 *      Returns the ConnectionID (1-8) of the current default file server.
 *      The ConnectionID refers to the ordinal position within the
 *      Connection ID Table ( found with fn_connID() ).
 *
 *      See the discussion under fn_pfConID() for a more detailed
 *      description of the preferred, default and primary server.
 *
 *  $EXAMPLES$
 *
 *      nDefServer := fn_defCID()
 *
 *  $SEEALSO$
 *      fn_pfConID()
 *  $INCLUDE$
 *
 *  $END$
 */

function fn_defCID()
  return _fnconnid( IDGET, GET_DEFAULT )

/*  $DOC$
 *  $FUNCNAME$
 *      fn_pfConID()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      GetPreferredConnectionID
 *  $SYNTAX$
 *
 *      fn_pfConID() -> nConnectionID
 *
 *  $ARGUMENTS$
 *
 *      None
 *
 *  $RETURNS$
 *
 *      The Connection ID of the preferred file server, or 0 if the
 *      preferred server is not set.
 *
 *  $DESCRIPTION$
 *
 *      This function determines the currently set preferred server.  The
 *      preferred server is the default server to which request packets are
 *      sent.  Whenever the shell constructs a request packet, if it does
 *      not have enough information to determine which file server should
 *      receive the packect (i.e., a drive specification), it will direct
 *      the packets to the current preferred file server.  If the preferred
 *      file server is 0 (unspecified), the packets will be directed to the
 *      file server that services the current default drive.
 *
 *      NOTE:  The preferred Connection ID is reset to 0 (unspecified) by
 *      the shell when an End of Job (EOJ) is done for a terminating
 *      process, or the EOJ is made.
 *
 *
 *      
 *      Preferred, Default and Primary Servers
 *      
 *
 *      NetWare uses the terms default, preferred and primary to prioritize
 *      the servers that a user is logged in to.  The question they deal
 *      with is essentially, "What server gets the information I'm sending
 *      now?"
 *
 *      The preferred server bears the highest priority for shell requests.
 *      A preferred server must be explicitly appointed by the workstation
 *      with fn_sPfCID().  Once appointed, that server receives all
 *      requests until the preferred server is changed.
 *
 *      If there is no preferred server, the default server is mapped with
 *      the default drive.  For example, if the default drive is F: and
 *      drive F: is mapped to FS1, _AND_ there is NO preferred server
 *      defined, the file server FS1 is the default server.
 *
 *      If the current drive is a local drive, the primary server is used.
 *      The primary server is the lowest conneciton priority of the 3
 *      server types and is usually the server the shell first attached to
 *      when it was started.  The primary server is used only if the
 *      default drive is set to a local drive and there is no preferred
 *      file server.
 *
 *  $EXAMPLES$
 *
 *      nPrefSvr := fn_pfConID()                  // determine pref'rd server
 *
 *  $SEEALSO$
 *      fn_connID() fn_sPfConID() fn_prConID() fn_defCID fn_EOJ()
 *  $INCLUDE$
 *
 *  $END$
 */

function fn_pfConID()
  return _fnconnid( IDGET, GET_PREFERRED )


/*  $DOC$
 *  $FUNCNAME$
 *      fn_sPfCID()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      SetPreferredConnectionID
 *  $SYNTAX$
 *
 *      fn_sPfCID( [ <nConnectionID> ] ) -> NIL
 *
 *  $ARGUMENTS$
 *
 *      <nConnectionID> is the Connection ID (1 - 8) for the server to be
 *      set as the default server to which request packets are sent.  If
 *      omitted, <nConnectionID> defaults to 0 (unspecified).
 *
 *  $RETURNS$
 *
 *      NIL
 *
 *  $DESCRIPTION$
 *
 *      This function sets the preferred server (1 - 8 ).  The preferred
 *      server is the default server to which request packets are sent.
 *
 *      If no preferred server is desired, set it to 0.
 *
 *      See the discussion under fn_pfConID() for a more detailed
 *      description of the preferred, default and primary server.
 *
 *  $EXAMPLES$
 *
 *      /* set server "SALES" to be the preferred server */
 *
 *      fn_sPfCID( ASCAN( fn_FSName(), "SALES" ) )
 *
 *  $SEEALSO$
 *      fn_pfConID()
 *  $INCLUDE$
 *
 *  $END$
 */

function fn_sPfCID( nConnID )
  return _fnconnid( IDSET, SET_PREFERRED, nConnID )


/*  $DOC$
 *  $FUNCNAME$
 *      fn_prConID()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      GetPrimaryConnectionID
 *  $SYNTAX$
 *
 *      fn_prConID() -> nConnectionID
 *
 *  $ARGUMENTS$
 *
 *      None
 *
 *  $RETURNS$
 *
 *      The Connection ID of the primary file server (1 - 8), or 0 if the
 *      primary file server is not set.
 *
 *  $DESCRIPTION$
 *
 *      This function determines the currently set primary server.  The
 *      primary file server is usually the first file server the user
 *      logs into.
 *
 *      See the discussion under fn_pfConID() for a more detailed
 *      description of the preferred, default and primary server.
 *
 *  $EXAMPLES$
 *
 *      nPrimSvr := fn_prConID()                  // determine primary server
 *
 *  $SEEALSO$
 *      fn_sPrCID() fn_pfConID()
 *  $INCLUDE$
 *
 *  $END$
 */

function fn_prConID()
  return _fnconnid( IDGET, GET_PRIMARY )


/*  $DOC$
 *  $FUNCNAME$
 *      fn_sPrCID()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      SetPrimaryConnectionID
 *  $SYNTAX$
 *
 *      fn_sPrCID( [ <nConnectionID> ] ) -> NIL
 *
 *  $ARGUMENTS$
 *
 *      <nConnectionID> is the Connection ID (1 - 8) for the server to be
 *      set as the primary.
 *
 *  $RETURNS$
 *
 *      Nothing
 *
 *  $DESCRIPTION$
 *
 *      This function sets the primary (1 - 8 ).  The primary server is
 *      the first server to which the user logs in.
 *
 *      If no primary server is desired, set it to 0.
 *
 *      See the discussion under fn_pfConID() for a more detailed
 *      description of the preferred, default and primary server.
 *
 *  $EXAMPLES$
 *
 *      /* set server "FSMAIN" to be the primary server */
 *
 *      fn_attchFS( "FSMAIN" )
 *
 *      fn_sPfCID( ASCAN( fn_FSName(), "FSMAIN" ) )
 *
 *  $SEEALSO$
 *      fn_prConID() fn_pfConID()
 *  $INCLUDE$
 *
 *  $END$
 */

function fn_sPrCID( nConnID )
  return _fnconnid( IDSET, SET_PRIMARY, nConnID )


/* internal function used by the primary/default/preferred functions above */

static function _fnconnid( xOp, xFunc, nID )
  local aRegs[ INT86_MAX_REGS ]

  default nID to 0

  aRegs[ AX ] := makehi( 240 ) + xFunc          // F0h

  if xOp == IDSET
     aRegs[ DX ] := nID
  endif

  if !ft_int86( INT21, aRegs )
     _fnSetErr( EINT86 )
  endif

return iif( xOp == IDGET, ( UNSIGNED( LowByte( aRegs[ AX ] ) ) ), nil )
